import 'rxjs/add/operator/switchMap';
import { Component, OnInit } from '@angular/core';
import { ActivatedRoute, Params } from '@angular/router';
import { Location } from '@angular/common';
import { FormGroup, FormControl, FormBuilder, Validators } from '@angular/forms';
import { NgbDateParserFormatter } from '@ng-bootstrap/ng-bootstrap';

import { User } from '../models/user';
import { Right } from '../../rights/models/right';
import { UserService } from '../services/user.service'
import { RightService } from '../../rights/services/right.service'

@Component({
  selector: 'app-user',
  templateUrl: './user.component.html',
  styleUrls: ['./user.component.less']
})
export class UserComponent implements OnInit {
  user: User;
  editMode: boolean;
  title: string;
  rights: any[];
  userForm: FormGroup;
  // = new FormGroup({
  //   firstName: new FormControl(),
  //   lastName: new FormControl(),
  //   userName: new FormControl(),
  //   birthDate: new FormControl(),
  //   rights: new FormControl(),
  //   });

  constructor(
    private userService: UserService,
    private rightService: RightService,
    private route: ActivatedRoute,
    private location: Location,
    private parserFormatter: NgbDateParserFormatter,
    private fb: FormBuilder

  ) { }

  selectedRight(selected): void {
    this.user.rights = this.user.rights || [];
    this.user.rights.push(selected);
  }

  removedRight(selected): void {
    this.user.rights = this.user.rights || [];
    var found = this.user.rights.find(u => u.id == selected.id);
    var index = this.user.rights.indexOf(found);
    if (index > -1)
      this.user.rights.splice(index, 1);
  }

  buildForm(): void {
    this.userForm = this.fb.group({
      'firstName': [this.user.firstName, [Validators.required]],
      'lastName': [this.user.lastName, [Validators.required]],
      'userName': [
        this.user.userName, [
          Validators.required,
          Validators.minLength(4),
          Validators.maxLength(24)
        ]
      ],
      'birthDate': [this.user.birthDate, [Validators.required]],
      'rights': [this.user.rights],
    });
    this.userForm.valueChanges
      .subscribe(data => this.onValueChanged(data));

    this.onValueChanged();
  }
  onValueChanged(data?: any) {
    if (!this.userForm) { return; }
    const form = this.userForm;

    for (const field in this.formErrors) {
      // clear previous error message (if any)
      this.formErrors[field] = '';
      const control = form.get(field);

      if (control && control.dirty && !control.valid) {
        const messages = this.validationMessages[field];
        for (const key in control.errors) {
          this.formErrors[field] += messages[key] + ' ';
        }
      }
    }
  }
  formErrors = {
    'firstName': '',
    'lastName': '',
    'userName': '',
    'birthDate': '',
    'rights': '',
  };

  validationMessages = {
    'firstName': {
      'required': 'Firstname is required.',
    },
    'lastName': {
      'required': 'Lastname is required.',
    },
    'userName': {
      'required': 'Username is required.',
      'minlength': 'Username must be at least 4 characters long.',
      'maxlength': 'Username cannot be more than 24 characters long.',
    },
    'birthDate': {
      'required': 'Birth date is required.',
    },
    'rights': {}
  };

  ngOnInit(): void {
    this.rightService.getRightsForCombo().then(rights => this.rights = rights);
    this.user = new User();
    this.buildForm();

    this.editMode = true;
    this.route.params.subscribe(params => {
      if (params['id']) {
        this.userService.getUser(params['id'])
          .then(user => {
            this.user = user
            this.user.rights = this.user.rights || [];
            this.user.rightsDisplay = this.user.rights.map(r => r.text).join(', ');
          });
      }
      if (this.location.path().indexOf('details') > -1) {
        this.editMode = false;
        this.title = "Details";
      }
      else {
        if (params['id'])
          this.title = "Edit";
        else {
          this.title = "Create";
          this.user = new User();
          this.user.rights = this.user.rights || [];
        }
      }
    });
  }

  save(): void {
    if (this.user.id)
      this.userService.updateUser(this.user)
        .then(() => this.goBack());
    else
      this.userService.createUser(this.user)
        .then(() => this.goBack());
  }

  goBack(): void {
    this.location.back();
  }
}
